//    Copyright (C) 2006 Paul Harrison
//    This program is free software; you can redistribute it and/or modify
//    it under the terms of the GNU General Public License as published by
//    the Free Software Foundation; either version 2 of the License, or
//    (at your option) any later version.
//
//    This program is distributed in the hope that it will be useful,
//    but WITHOUT ANY WARRANTY; without even the implied warranty of
//    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//    GNU General Public License for more details.
//
//    You should have received a copy of the GNU General Public License
//    along with this program; if not, write to the Free Software
//    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA



//==============================================================================
// The parameters of the SVG itself.
//==============================================================================

var svg, width, height;  // The top-level SVG object and its dimensions

//==============================================================================
// The following triplets define parameters specific to the six- and four-sided
// versions of the tiles. These are assigned to connections, xMapper, and
// yMapper whenever the current tileset changes.
//
// Hexagonal connection pattern:
//
//     4 5
//   3 * 0
//   2 1
//
// (all points remain on a square grid, but a regular hexagon pattern
//  can be formed by a simple linear transformation)
//
// Square connection pattern:
//
//    3
//  2 * 0
//    1
//
//==============================================================================

var	connectionsHex = [ [ 1,  0,  3],   // Lookup table for hexagonal tiles. Each
				      [ 0,  1,  4],   // element corresponds to a neighboring tile
					  [-1,  1,  5],   // in standard index order and contains the
					  [-1,  0,  0],   // x and y offsets relative to the tile and
					  [ 0, -1,  1],   // the index of the reverse connection.
					  [ 1, -1,  2] ];
var xMapperHex     = [ 1.0, 0.0 ];              // linear transform for x coords
var yMapperHex     = [ 0.5, Math.sqrt(0.75) ];  // linear transform for y coords

var	connectionsQuad = [ [ 1,  0,  2],    // As above, but for square tiles.
					  [ 0,  1,  3],
					  [-1,  0,  0],
					  [ 0, -1,  1] ];
var xMapperQuad     = [ 1.0, 0.0 ];
var yMapperQuad     = [ 0.0, 1.0 ];


//==============================================================================
// Connector definitions. In the original Ghost Diagrams, these were a fixed
// set. In the current version of Diagrammata, they are mutable, though the
// defaults currently correspond to the original. [WORK IN PROGRESS]
// 
// The properties of each connector are as follows:
//
//		mate ... the connector of the opposite polarity or, for non-polar
//               connectors, itself. A null value indicates no mate.
//      visible ... whether the connector is visible
//      polarity ... -1 negative, 0 neutral, +1 positive
//      pokiness ... deviation from flatness for drawing routines
//      edgeWidth ... weight of edges in drawing routines
//      weight ... weighting for random selection
//
//==============================================================================

var dcon = {

	// Non-polar connectors

	"0": { mate:  "0", visible:  true, polarity:  0, pokiness:  0.00, edgeWidth: 0.150, weight: 1 },
	"1": { mate:  "1", visible:  true, polarity:  0, pokiness:  0.00, edgeWidth: 0.400, weight: 1 },
	"2": { mate:  "2", visible:  true, polarity:  0, pokiness:  0.00, edgeWidth: 0.300, weight: 1 },
	"3": { mate:  "3", visible:  true, polarity:  0, pokiness:  0.00, edgeWidth: 0.225, weight: 1 },
	"4": { mate:  "4", visible:  true, polarity:  0, pokiness:  0.00, edgeWidth: 0.150, weight: 0 },
	"5": { mate:  "5", visible:  true, polarity:  0, pokiness:  0.00, edgeWidth: 0.400, weight: 0 },
	"6": { mate:  "6", visible:  true, polarity:  0, pokiness:  0.00, edgeWidth: 0.300, weight: 0 },
	"7": { mate:  "7", visible:  true, polarity:  0, pokiness:  0.00, edgeWidth: 0.225, weight: 0 },
	"8": { mate:  "8", visible:  true, polarity:  0, pokiness:  0.00, edgeWidth: 0.150, weight: 0 },
	"9": { mate:  "9", visible:  true, polarity:  0, pokiness:  0.00, edgeWidth: 0.400, weight: 0 },

	// Polar connectors

	"A": { mate:  "a", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.400, weight: 1 },
	"a": { mate:  "A", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.400, weight: 1 },
	"B": { mate:  "b", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.300, weight: 1 },
	"b": { mate:  "B", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.300, weight: 1 },
	"C": { mate:  "c", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.225, weight: 1 },
	"c": { mate:  "C", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.225, weight: 1 },
	"D": { mate:  "d", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.150, weight: 1 },
	"d": { mate:  "D", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.150, weight: 1 },
	"E": { mate:  "e", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.400, weight: 0 },
	"e": { mate:  "E", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.400, weight: 0 },
	"F": { mate:  "f", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.300, weight: 0 },
	"f": { mate:  "F", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.300, weight: 0 },
	"G": { mate:  "g", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.225, weight: 0 },
	"g": { mate:  "G", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.225, weight: 0 },
	"H": { mate:  "h", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.150, weight: 0 },
	"h": { mate:  "H", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.150, weight: 0 },
	"I": { mate:  "i", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.400, weight: 0 },
	"i": { mate:  "I", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.400, weight: 0 },
	"J": { mate:  "j", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.300, weight: 0 },
	"j": { mate:  "J", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.300, weight: 0 },
	"K": { mate:  "k", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.225, weight: 0 },
	"k": { mate:  "K", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.225, weight: 0 },
	"L": { mate:  "l", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.150, weight: 0 },
	"l": { mate:  "L", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.150, weight: 0 },
	"M": { mate:  "m", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.400, weight: 0 },
	"m": { mate:  "M", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.400, weight: 0 },
	"N": { mate:  "n", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.300, weight: 0 },
	"n": { mate:  "N", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.300, weight: 0 },
	"O": { mate:  "o", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.225, weight: 0 },
	"o": { mate:  "O", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.225, weight: 0 },
	"P": { mate:  "p", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.150, weight: 0 },
	"p": { mate:  "P", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.150, weight: 0 },
	"Q": { mate:  "q", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.400, weight: 0 },
	"q": { mate:  "Q", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.400, weight: 0 },
	"R": { mate:  "r", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.300, weight: 0 },
	"r": { mate:  "R", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.300, weight: 0 },
	"S": { mate:  "s", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.225, weight: 0 },
	"s": { mate:  "S", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.225, weight: 0 },
	"T": { mate:  "t", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.150, weight: 0 },
	"t": { mate:  "T", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.150, weight: 0 },
	"U": { mate:  "u", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.400, weight: 0 },
	"u": { mate:  "U", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.400, weight: 0 },
	"V": { mate:  "v", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.300, weight: 0 },
	"v": { mate:  "V", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.300, weight: 0 },
	"W": { mate:  "w", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.225, weight: 0 },
	"w": { mate:  "W", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.225, weight: 0 },
	"X": { mate:  "x", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.150, weight: 0 },
	"x": { mate:  "X", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.150, weight: 0 },
	"Y": { mate:  "y", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.400, weight: 0 },
	"y": { mate:  "Y", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.400, weight: 0 },
	"Z": { mate:  "z", visible:  true, polarity:  1, pokiness:  0.15, edgeWidth: 0.300, weight: 0 },
	"z": { mate:  "Z", visible:  true, polarity: -1, pokiness: -0.15, edgeWidth: 0.300, weight: 0 },

	// Invisible connectors

	"-": { mate:  "-", visible: false, polarity:  0, pokiness:  0.00, edgeWidth: 0.000, weight: 1 },
	"_": { mate: null, visible: false, polarity:  0, pokiness:  0.00, edgeWidth: 0.000, weight: 2 },
};

var polarPair = [ ]; // list of polar pairs generated in init() and used in testCharge()

var polarCount    = 4;
var nonpolarCount = 4;


//==============================================================================
// Tile colors, applied sequentially with rollover. At present, this rarely if
// ever happens because the tile set would have to have 18 or more tiles in it.
//==============================================================================

var colors  = [ 
	"#D4C56A", "#554800", "#133253", "#2B4C6F", "#801F15", "#FFB2AA", "#805515", 
	"#FFF2AA", "#D4746A", "#553300", "#550800", "#041D37", "#807015", "#D4AA6A", 
	"#4A6A8A", "#FFDDAA", "#738CA6"
];

//==============================================================================
// Startup defaults. These get assigned to the hex grid defaults at startup.
//==============================================================================

var connections      = connectionsHex;  // current connection map
var xMapper          = xMapperHex;      // current x linear transform
var yMapper          = yMapperHex;      // current y linear transform
var sides            = 6;               // current number of tile sides
var tileSize         = 12.0;            // current scaling value
var drawTilesColored = false;           // toggles tile coloring
var drawTilesKnotted = false;           // toggles knotted rendering style
var prec             = 6;               // digits of precision to use to suppress (most) floating point errors in SVG welding

//==============================================================================
// Advanced parameters
//==============================================================================

var debugging     = true;     // if true, enables debugging output

var maxIter       = 0;        // if nonzero, halts execution when clock == maxIter
var rotation      = 0;        // rotation of whole configuration in degrees
var showClock     = false;    // if true, display current clock value
var useTestCharge = true;     // enforce testCharge during randomization
var fixedSides    = "random"; // number of sides in random generation: random, 4, or 6 
var baseTileCount = 2;        // base number of tiles in random generation
var anyBadTest    = true;     // use the anyBad filter

//==============================================================================
// State variables for current run.
//==============================================================================

var tileDefs;          // an array of all possible rotations of all tiles in the current set
var tileColors;        // object mapping tile strings to colors
var tiles;             // object mapping location keys to tile instances, which may have been rotated
var nTiles;			   // number of tiles currently in place
var clock;			   // counter incremented every time a tile is placed
var tileNodeCache;     // object mapping tile strings to precalculated SVG elements
var times;             // map of location keys to the clock value when a tile was last placed there; used in backtracking
var todo;              // dictionary of location keys for which tile placement is pending
var badSignatures;     // dictionary of bad signatures
var badSignaturesSize; // count of elements in badSignatures; used in lieu of badSignatures.length for no apparent reason
var optionCache;       // object mapping position keys to legal tiles that could be placed in them (?)
var recentlyRemoved;   // dictionary of positions from which a tile has been recently removed
var nToRemove;         // number of tiles to delete during backtracking
var whereToRemove;     // start location for backtracking
var intervalId = null; // timer interval for calls to iterate()
var randomizing;       // flag indicating randomization is in progress

var uiState        = null; 
var lastTileString = "";

//==============================================================================
// State for SVG welding routine
//==============================================================================

var welder = {
	frag:       null,
	subpath:    null,
	offset:     null,
	initLength: null,
	timerId:    null,
	jobSize:    null,  // number of fragments to attempt in an iteration
	refresh:     500,  // target length of iteration in ms
	output:     null,
}


//##############################################################################
//# General utility functions.
//##############################################################################

//==============================================================================
// Returns true if value is an element of list, false otherwise.
//==============================================================================

function listContains(list, value) {
	for(var i = 0; i < list.length; i++)
		if(list[i] == value)
			return true;
	return false;
}


//==============================================================================
// Returns a random positive integer less than n.
//==============================================================================

function randomInt(n) {
	return Math.floor(Math.random()*n);
}


//==============================================================================
// Returns a randomly chosen element from the supplied array.
//==============================================================================

function randomChoice(list) {
	return list[randomInt(list.length)];
}


//##############################################################################
//# Point [x,y] operations
//##############################################################################

function add(a,b) {
	return [ a[0] + b[0], a[1] + b[1] ];
}

function sub(a,b) {
	return [a[0]-b[0],a[1]-b[1]];
}

function scale(point, factor) {
	return [ point[0] * factor, point[1] * factor ];
}

function left90(point) {
	return [ point[1], -point[0] ];
}


//==============================================================================
// Calculates the distance of the point from the origin.
//==============================================================================

function length(point) {
	return Math.sqrt(point[0] * point[0] + point[1] * point[1]);
}


//==============================================================================
// Returns a string with the point coordinates in the form "x y".
//==============================================================================

function render(point) {
	return point[0] + " " + point[1];
}


//##############################################################################
//# Position [x,y] operations
//##############################################################################

//==============================================================================
// Given tile coordinates, return screen coordinates.
//==============================================================================

function transform(position) {
	return scale(add(scale(xMapper, position[0]), scale(yMapper, position[1])), tileSize);
}


//==============================================================================
// Converts a position [x, y] into a hash key. This is used as an index in
// several lookup tables as well as for SVG tile element IDs.
//==============================================================================

function toKey(position) {
	return ((position[0] + 0x4000) << 15) + (position[1] + 0x4000);
}


//==============================================================================
// Converts a hash key back into a position [x, y].
//==============================================================================

function fromKey(key) {
	return [(key >> 15) - 0x4000, (key & 0x7fff) - 0x4000];
}


//##############################################################################
//# Returns a serialized representation of the currently placed tiles. This
//# consists of a semicolon-delimited list of first all of the location keys,
//# then all of the tile strings corresponding to them. Used in iterate,
//# apparently to keep track of dead-end configurations. [?]
//##############################################################################

function signature() {
    var list = [];
    for(var key in tiles) {
        list.push(key);
    }
    list.sort();

    var n = list.length;
    for(var i = 0; i < n; i++)
        list.push(tiles[list[i]]);

    return list.join(';')
}


//##############################################################################
//# For a given position, return the list of tiles that could be placed there.
//##############################################################################

function options(position) {
    
	var key = toKey(position);

	if(optionCache[key] != null) 
		return optionCache[key];

	//==========================================================================
	// Construct a suitable regex to filter the available possibilities.
	//==========================================================================

    var pattern = '^';
    
	for(var i = 0; i < sides; i++) {
    
		var neighbor = tiles[toKey(add(position, connections[i]))];     // connections[offset] = [ x, y, reverse_connector ];
       
		//----------------------------------------------------------------------
		// If the neighboring position is empty, anything will do. Otherwise,
		// look up the matching connector.
		//----------------------------------------------------------------------

		if(neighbor == null)
			pattern += '.';
		else if(dcon[neighbor[connections[i][2]]].mate === null)
			pattern += neighbor[connections[i][2]];
		else
			pattern += dcon[neighbor[connections[i][2]]].mate;
    }
    
	pattern += '$';
    regexp = new RegExp(pattern);

    optionCache[key] = tileDefs.filter( function(x) { 
		return regexp.test(x); 
	});

    return optionCache[key];
}


//##############################################################################
//# Places the tile in the specified position.
//##############################################################################

function put(position, tile) {
    var key = toKey(position);

    draw(position, tile);

    for(var i = 0; i < sides; i++) {
        var nPosition = add(position, connections[i]);
        var nKey = toKey(nPosition);
        if(dcon[tile[i]].mate !== null) {
            if(tiles[nKey] == null && isVisible(nPosition))
                todo[nKey] = true;
        }
        if(optionCache[nKey] != null)
            delete optionCache[nKey];
    }

    tiles[key] = tile;
    times[key] = clock;
    delete todo[key];
    clock++;
	if(showClock)
		updateClock();
    nTiles++;

	if(maxIter > 0 && clock >= maxIter)
		stop();

}


//##############################################################################
//# Removes the tile from the specified position. Also appears to tidy up
//# pending child positions.
//##############################################################################

function remove(position) {
    var key = toKey(position);

    delete tiles[key];
    nTiles -= 1;
    todo[key] = true;
    recentlyRemoved[key] = true;

    svg.removeChild(document.getElementById(key));

    for(var i=0; i < sides; i++) {
        var nPosition = add(position, connections[i]);
        var nKey = toKey(nPosition);
        if(todo[nKey] && !isValidTodo(nPosition))
            delete todo[nKey];
        if(optionCache[nKey])
            delete optionCache[nKey];
    }
}


//##############################################################################
//# Returns true if the supplied position has at least one neighboring tile
//# whose back connector is not '_' -- in other words, whether this is a legal
//# position to add a new tile.
//##############################################################################

function isValidTodo(position) {

    for(var i = 0; i < sides; i++) {
		var neighbor = tiles[toKey(add(position, connections[i]))];

		if(neighbor !== undefined && dcon[neighbor[connections[i][2]]].mate !== null)
			return true;
    }

    return false;
}


//##############################################################################
//# Tests whether a tile can be removed from the specified position. Returns
//# true if none of the connected neighbors are more recent than the current
//# tile.
//##############################################################################

function canRemove(position) {
    var key = toKey(position);
    if(tiles[key] == null) return false;

    for(var i=0; i < sides; i++) {
        var nPosition = add(position,connections[i]);
        var nKey = toKey(nPosition);

        if(tiles[nKey] != null && times[nKey] > times[key])
            return false;

    }

    return true;
}


//##############################################################################
//# Called once in the main loop. Attempts to delete the tile at the specified
//# position or, failing that, its nearest deletable neighbor.
//##############################################################################

function removeNear(position) {

	var queue  = [ toKey(position) ];         // Create a queue consisting initially
    var queued = {};                          // of only the supplied position.

    for(var i = 0; i < queue.length; i++) {

		//----------------------------------------------------------------------
		// If the current queue item can be removed, remove it and exit.
		//----------------------------------------------------------------------

        var position2 = fromKey(queue[i]);
        if(canRemove(position2)) {
            remove(position2);
            return;
        }

		//----------------------------------------------------------------------
		// Otherwise, starting at a random edge, add any existing neighbor to
		// the queue that hasn't been queued already or recently deleted.
		//----------------------------------------------------------------------

        var start = randomInt(sides);
        var j     = start;

        while(true) {
            var nPosition = add(position2,connections[j]);
            var nKey = toKey(nPosition);
            if(!queued[nKey] && (recentlyRemoved[nKey] || tiles[nKey] != null)) {
                queue.push(nKey);
                queued[nKey] = true;
            }

            j = (j+1) % sides;
            if(j == start)
				break;
        }
    }
}


//##############################################################################
//# Generates the actual tile paths. This is the whole tile in the case of
//# the regular style, and a subcomponent in the case of the knotted style.
//# Returns the resulting SVG element.
//##############################################################################

function makeShape(tile, thickness) {

    var connectors = [ ];

	for(var i = 0; i < tile.length; i++) {

		if(dcon[tile[i]] == undefined || dcon[tile[i]].mate === null || dcon[tile[i]].visible === false)
			continue;

        var out	      = transform([ connections[i][0] * 0.5, connections[i][1] * 0.5 ]);
        var left      = left90(out);
        var pokiness  = dcon[tile[i]].pokiness;
		var edgeWidth = dcon[tile[i]].edgeWidth;

        var a         = add(out, scale(left, edgeWidth));
        var b         = scale(out, 1.0 + pokiness);
        var c         = add(out, scale(left, -edgeWidth));

        var aShrunk   = add(scale(a, thickness), scale(b, 1.0 - thickness));
        var cShrunk   = add(scale(c, thickness), scale(b, 1.0 - thickness));

        connectors.push([ a, b, c, out, aShrunk, cShrunk ]);
    }

    if(connectors.length == 1) {
        var point = scale(connectors[0][3], -0.3);
        var out   = scale(left90(connectors[0][3]), 1.25);

        connectors.push([ point, point, point, out, point, point ]);
        out = scale(out, -1.0);
        connectors.push([ point, point, point, out, point, point ]);
    }

    var last = connectors[connectors.length-1];

    var d = 'M ' + render(last[5]);
    for(var i = 0; i < connectors.length; i++) {
        var scaler = length(sub(connectors[i][0],last[2])) / tileSize;
        //if(connectors.length == 1) scaler *= 3;

        d += ' C ' + render(sub(last[5],scale(last[3],scaler))) +
               ' ' + render(sub(connectors[i][4],scale(connectors[i][3],scaler))) +
               ' ' + render(connectors[i][4]) +
             ' L ' + render(connectors[i][1]) +
             ' L ' + render(connectors[i][5]);
        last = connectors[i];
    }
    d += ' z';

    var element = document.createElementNS('http://www.w3.org/2000/svg', 'path');
    element.setAttributeNS(null, 'd', d);
    return element;
}

/*

//##############################################################################
//# Generates the actual tile paths. This is the whole tile in the case of
//# the regular style, and a subcomponent in the case of the knotted style.
//# Returns the resulting SVG element.
//##############################################################################

function makeShape(tile, thickness) {
    var connectors = [ ];
    for(var i = 0; i < sides; i++) {
        if(tile[i] == '_' || tile[i] == '-') continue;

        var out = transform([ connections[i][0] * 0.5, connections[i][1] * 0.5 ]);
        var left = left90(out);

        var pokiness, edge_width, curviness;

        if(tile[i] == 'A' || tile[i] == 'B' || tile[i] == 'C' || tile[i] == 'D')
            pokiness = 0.15;
        else if(tile[i] == 'a' || tile[i] == 'b' || tile[i] == 'c' || tile[i] == 'd')
            pokiness = -0.15;
        else
            pokiness = 0.0;

        if(tile[i] == 'A' || tile[i] == 'a' || tile[i] == '1')
            edge_width = 0.4;
        else if(tile[i] == 'B' || tile[i] == 'b' || tile[i] == '2')
            edge_width = 0.3;
        else if(tile[i] == 'C' || tile[i] == 'c' || tile[i] == '3')
            edge_width = 0.225;
        else
            edge_width = 0.15;

        var a = add(out, scale(left, edge_width));
        var b = scale(out, 1.0 + pokiness);
        var c = add(out, scale(left, -edge_width));

        var a_shrunk = add(scale(a, thickness), scale(b, 1.0 - thickness));
        var c_shrunk = add(scale(c, thickness), scale(b, 1.0 - thickness));

        connectors.push([ a, b, c, out, a_shrunk, c_shrunk ]);
    }

    if(connectors.length == 1) {
        var point = scale(connectors[0][3], -0.3);
        var out = scale(left90(connectors[0][3]), 1.25);
        connectors.push([ point,point,point, out, point,point ]);
        out = scale(out, -1.0);
        connectors.push([ point,point,point, out, point,point ]);
    }

    var last = connectors[connectors.length-1];
    var d = 'M ' + render(last[5]);
    for(var i = 0; i < connectors.length; i++) {
        var scaler = length(sub(connectors[i][0],last[2])) / tileSize;
        //if(connectors.length == 1) scaler *= 3;

        d += ' C ' + render(sub(last[5],scale(last[3],scaler))) +
               ' ' + render(sub(connectors[i][4],scale(connectors[i][3],scaler))) +
               ' ' + render(connectors[i][4]) +
             ' L ' + render(connectors[i][1]) +
             ' L ' + render(connectors[i][5]);
        last = connectors[i];
    }
    d += ' z';

    var element = document.createElementNS('http://www.w3.org/2000/svg', 'path');
    element.setAttributeNS(null, 'd', d);
    return element;
}


*/


//##############################################################################
//# Returns SVG element for the specified tile string in the simple style.
//##############################################################################

function makeTileNode(tile) {
    var element = makeShape(tile, 1.0);

    element.setAttributeNS(null, 'fill', drawTilesColored ? tileColors[tile] : 'black');
    if(drawTilesColored) {
        element.setAttributeNS(null, 'stroke', 'black');
        element.setAttributeNS(null, 'stroke-width', '1');
    }

    return element;
}


//##############################################################################
//# Returns SVG element for the specified tile string in the knotted style.
//##############################################################################

function makeKnottedTileNode(tile) {
    var element = document.createElementNS('http://www.w3.org/2000/svg', 'g');
    var subElement;

    var n = 0;
    for(var i = 0; i < sides; i++)
		if(dcon[tile[i]].visible && dcon[tile[i]].mate !== null) 
			n += 1;

    var subTiles;
    if(n < 4) {
        subTiles = [ tile ];
    } else {
        subTiles = [ tile, tile, tile ];
        for(var i = 0; i < sides; i++) {
			if(dcon[tile[i]].polarity != -1)
                subTiles[0] = subTiles[0].slice(0,i) + ' ' + subTiles[0].slice(i+1);   // IMPORTANT: the spaces here and below
			if(dcon[tile[i]].polarity != 1)                                            // are NOT connector codes!
                subTiles[1] = subTiles[1].slice(0,i) + ' ' + subTiles[1].slice(i+1);
			if(!(dcon[tile[i]].polarity == 0 && dcon[tile[i]].visible))
                subTiles[2] = subTiles[2].slice(0,i) + ' ' + subTiles[2].slice(i+1);
        }
    }

    for(var i = 0; i < subTiles.length; i++)
        if(subTiles[i] != '      ' && subTiles[i] != '    ') {
            subElement = makeShape(subTiles[i], 1.2);
            subElement.setAttributeNS(null, 'fill', 'black');
            element.appendChild(subElement);

            subElement = makeShape(subTiles[i], 0.8);
            subElement.setAttributeNS(null, 'fill', drawTilesColored ? tileColors[tile] : '#00c9ed');
            subElement.setAttributeNS(null, 'stroke', drawTilesColored ? tileColors[tile] : '#00c9ed');
            subElement.setAttributeNS(null, 'stroke-width', '1.0');
            element.appendChild(subElement);
        }

    return element
}

//##############################################################################
//# Returns SVG element for the specified tile string, handing off the actual
//# work to makeKnottedTileNode or makeTileNode depending on the current
//# drawing mode.
//##############################################################################

function getTileNode(tile) {
    if(tileNodeCache[tile] != null)
        return tileNodeCache[tile];

    var element = drawTilesKnotted ? makeKnottedTileNode(tile) : makeTileNode(tile);
    tileNodeCache[tile] = element;
    return element;
}


//##############################################################################
//# Draws an individual tile.
//##############################################################################

function draw(position, tile) {
	var element = getTileNode(tile).cloneNode(true);
	var translation = transform(position);
	element.setAttributeNS(null, 'id', toKey(position));
	element.setAttributeNS(null, 'transform', 'translate(' + (translation[0] + width / 2) + ',' + (translation[1] + height / 2) + ')');
	svg.appendChild(element);
}


//##############################################################################
//# Returns true if position is within the viewport, false otherwise.
//##############################################################################

function isVisible(position) {
    var point = transform(position);
    var allowance = tileSize;
    return point[0] >= -width  / 2 - allowance && point[0] <= width  / 2 + allowance &&
           point[1] >= -height / 2 - allowance && point[1] <= height / 2 + allowance;
}


//##############################################################################
//# Main loop, called via timer interval initialized in init().
//##############################################################################

function iterate() {

	//==========================================================================
	// If we are randomizing and have placed ten or more tiles, the
	// randomization phase is over.
	//==========================================================================

    if(nTiles >= 10)
		randomizing = false;

	//==========================================================================
	// If we're still randomizing and haven't managed to place ten tiles by the
	// 1000th iteration, it's time to stop the current job and restart
	// randomization (which in turn calls init).
	//==========================================================================

    if(randomizing && clock > 1000) {
        stop();
        randomize();
        return;
    }

	//==========================================================================
	// Each iteration evidently involves 16 iterations of this loop. (FIXME:
	// this should be configurable through the UI.)
	//==========================================================================

    for(var i = 0; i < 16; i++) {

		//----------------------------------------------------------------------
		// If there are pending removals, we get rid of them first, with each
		// removal ending the loop.
		//----------------------------------------------------------------------

		if(nToRemove) {
            removeNear(whereToRemove);
            nToRemove -= 1;
            continue;
        }

		//----------------------------------------------------------------------
		// Loop through the pending placement list and find the position which
		// is both nearest the origin and has the fewest connection options(?)
		//----------------------------------------------------------------------

        var key       = null;
        var keyScore  = 1e30;

		for(key2 in todo) {

            var position2  = fromKey(key2);
            var key2Score = length(transform(position2));

            if(options(position2).length < 2)
                key2Score -= 65536;

            if(key2Score < keyScore) {
                key       = key2;
                keyScore = key2Score;
            }

        }

		//----------------------------------------------------------------------
		// If that didn't leave us with a key, then we have reached the end of
		// the current tile configuration and call stop to terminate the
		// iteration interval OR, if we are randomizing, we've reached a dead
		// end and need to restart the randomization process.
		//----------------------------------------------------------------------

        if(key == null) {
            stop();
            if(randomizing) {
				randomize();
			} else {
				uiState = "finished";
			}
			uiUpdate();
            return;
        }

		//----------------------------------------------------------------------
		// Get the option list from our selected position and shuffle it.
		//----------------------------------------------------------------------

        var position    = fromKey(key);
        var optionList = options(position);

		for(var j = 0; j < optionList.length; j++) {

			var k    = randomInt(optionList.length);
            var temp = optionList[k];

			optionList[k] = optionList[j];
            optionList[j] = temp;
        }

		//----------------------------------------------------------------------
		// Attempt to place tiles at all of the open positions around the
		// current tile. If the resulting configuration is okay, bail out of
		// the loop. Otherwise, remove the bad tile and try the next option.
		//----------------------------------------------------------------------

        var j=0;
        while(j < optionList.length) {
            put(position, optionList[j]);

            if(!badSignatures[signature()])
				break;

            remove(position);
            j += 1;
        }

		//----------------------------------------------------------------------
		// This last block fires only if all attempts to place a tile failed.
		//----------------------------------------------------------------------

        if(j == optionList.length) { 

			// If this was the very first tile, then we've been in a dead end
			// from the very beginning. If we haven't exhausted the randomization
			// period, perform some randomization and bail out. (The randomize()
			// function will trigger a fresh init() call.)

			// If we're not randomizing, then we are done.

            if(nTiles == 1) {
                stop();
                if(randomizing) {
					randomize();
				} else {
					uiState = "finished";
				}
				uiUpdate();
                return;
            }

			// If we've accumulated 1024 (FIXME: make this configurable) known
			// bad configurations, flush the cache.

            if(badSignaturesSize >= 1024) {
                badSignatures = {};
                badSignaturesSize = 0;
            }

			// Add the current configuration to the cache of known bad
			// configurations

            badSignatures[signature()] = true;
            badSignaturesSize += 1;

			// Determine the extent of the backtrack

            nToRemove = 1;
            while(Math.random() < 1.0/4) {
				nToRemove *= 2;
			}
            if(nToRemove >= nTiles) {
				nToRemove = nTiles-1;
			}

			// Set the beginning of the backtracking to the current position
			// and flush the cache of recently removed tiles

            whereToRemove = position;
            recentlyRemoved = {};
        }
    }
}


//##############################################################################
//# Clears the tile node cache and SVG element, and then redraws. Called after
//# the drawing parameters have been changed.
//##############################################################################

function redraw() {
    tileNodeCache = {};
    svg.textContent = '';

    for(key in tiles)
		draw(fromKey(key), tiles[key]);
}


//##############################################################################
//# Generates a URL with the GET variable necessary to render the currently
//# active pattern and redirects the browser to it.
//##############################################################################

function makeURL() {
	var baseTiles = tilesetParse($("#textfield").val());
    location = document.URL.split('?')[0].split('#')[0] + '?' + baseTiles.join('&amp;');
}


//##############################################################################
//# Initializes data structures and starts rendering process. Any existing
//# process is stopped first.
//##############################################################################

function init() {

	//==========================================================================
	// Put a stop to any nonsense going on in the UI
	//==========================================================================

	stop();             // stop any existing run
    displayTileSize();  // update control to match current value of tileSize

	uiState = "running";
	uiUpdate();

	//==========================================================================
	// Do some massaging of the connector table, dcon, and derived values.
	//==========================================================================

	var tmp = { };
	for(var c in dcon) {
		if(tmp[c] === undefined && dcon[c].polarity != 0) {
			polarPair.push([c, dcon[c].mate]);
			tmp[c] = true;
			tmp[dcon[c].mate] = true;
		}
	}
	
	//==========================================================================
	// Initialize state variables
	//==========================================================================

    svg    = document.getElementById('svgOuter');
    width  = parseInt( $("#svgOuter").width() );
    height = parseInt( $("#svgOuter").height() );
    svg    = document.getElementById('svg');

    tiles               = {};
    nTiles              = 0;
    tileDefs            = [];
    tileColors          = {};
    times               = {};
    clock               = 0;
    todo                = {};
    badSignatures       = {};
    badSignaturesSize   = 0;
    optionCache         = {};
    recentlyRemoved     = {};
    nToRemove           = 0;
    randomizing         = false;

    redraw();                   // In this case, this serves to clear the screen

    todo[toKey([0,0])] = true;  // add origin to list of pending tile locations

	//==========================================================================
	// Grab the tileset from the UI, run it through eval, and check the length
	// of the first tile string to set up the necessary parameters.
	//==========================================================================

	var tileString = $("#textfield").val().replace(/^\s+/, "").replace(/\s+/, "").replace(/\s+/g, " ");
	if(tileString != lastTileString) {
		lastTileString = tileString;
		colors.sort(Math.random);
	}

    var baseTiles = tilesetParse(tileString);

    if(baseTiles[0].length == 6) {
        sides       = 6;
        connections = connectionsHex;
        xMapper     = xMapperHex;
        yMapper     = yMapperHex;
    } else if(baseTiles[0].length == 4) {
        sides       = 4;
        connections = connectionsQuad;
        xMapper     = xMapperQuad;
        yMapper     = yMapperQuad;
    } else {
		return;
	}

	// Check to make sure all tiles are the same length as the first -----------

    if(!baseTiles.every(function(x) { return x.length == sides; })) return;

	//--------------------------------------------------------------------------
	// Add the supplied tiles to tileDefs and assign tile colors. Filter out
	// duplicate tiles, including rotations.
	//--------------------------------------------------------------------------

    for(var i = 0; i < baseTiles.length; i++) {
        var tile = baseTiles[i];
        while(!listContains(tileDefs, tile)) {
            tileDefs.push(tile);
            tileColors[tile] = colors[i%colors.length];
            tile = tile.substring(1) + tile[0];
        }
    }

    put([0,0], tileDefs[0]);  // put first tile at origin (may be replaced later)

    intervalId = window.setInterval("iterate()", 10)  // FIXME: make interval configurable in UI
}


//##############################################################################
//# Parses a serialized tileset. Since we've replaced the original ' ' with '_',
//# the difference between the old and new formats can be handled with a simple
//# regex. Returns the resulting tileset as an array of strings.
//##############################################################################

function tilesetParse(str) {
	str = str.trim().replace(/'/g, '');
	return str.split(/\s*,\s*/)
}


//##############################################################################
//# Halts the assembling of the tiling if one is in progress.
//##############################################################################

function stop() {
    if(intervalId != null) {
        window.clearInterval(intervalId);
        intervalId = null;
    }
	if(svg)
		redraw();
}


//##############################################################################
//# This function rotates the characters in the tile until it finds the sequence
//# with the highest lexical sort order. All rotations of tile characters are
//# equivalent, so normalize() is called by randomize() before checking for
//# duplicate tiles in the generated set.
//##############################################################################

function normalize(tile) {
    var best = tile;
    for(var i = 1; i < tile.length; i++) {
        tile = tile.substring(1) + tile[0];
        if(tile > best) best = tile;
    }
    return best;
}


//##############################################################################
//# Generates a random set of tiles and begins drawing. This is easily the part
//# of the original code most open to improvement and which contains the great-
//# est number of parameters that would be interesting to tweak via the UI.
//##############################################################################

function randomize() {

	console.clear();
	debug("Entered randomize()");

	uiState = "running";
	uiUpdate();

	//==========================================================================
	// Definitely need to better understand the knotted tile routines. Here,
	// we're defaulting to a hexagonal grid if we're in knotted tile mode, and
	// choosing randomly between hexes and squares otherwise.
	//==========================================================================

	if(fixedSides == "random")
	    sides = drawTilesKnotted ? 6 : randomChoice([4,6]);
	else
		sides = fixedSides;

	//==========================================================================
	// The 'palette' generated here is a string composed of edge symbols that
	// are concatenated together with repetition serving the role of weighting.
	// (Later, when we draw from the palette, we use random indices into the
	// string.
	//==========================================================================

    var palette = '';

	//--------------------------------------------------------------------------
	// This currently mimics the original randomization procedure for blanks,
	// which are now invisible non-polar connectors. We construct an array of
	// blanks, with copies equal to the weight parameter in dcon, and then
	// select one of them at random. This is expanded into a string sides chars
	// long to test against randomly generated tiles to determine if the entire
	// tile is blank.
	//--------------------------------------------------------------------------

	var blank = [];
	for(var c in dcon) {
		if(dcon[c].visible == false && dcon[c].polarity == 0 && dcon[c].weight > 0) { 
			for(var i = 0; i < dcon[c].weight; i++) {
				blank.push(c);
			}
		}
	} 
	blank = randomChoice(blank);
	for(var i = 1; i < sides; i++)
		blank += blank[0];

	//--------------------------------------------------------------------------
	// Determine a random number of tiles based on a distribution with a norm
	// of 2.
	//--------------------------------------------------------------------------

    var tileCount = baseTileCount;
    while(Math.random() < 0.4)
		tileCount += 1;

	debug("tileCount set to " + tileCount);

	//--------------------------------------------------------------------------
	// Generate the contents of the palette. The size of the palette corresponds
	// to the variety of options.
	//--------------------------------------------------------------------------

	var cPairs = [ ];
	for(var c in dcon) {
		for(var i = 0; i < dcon[c].weight; i++) {
			if(dcon[c].visible) {
				cPairs.push(c.concat(dcon[c].mate));
			}
		}
	}

	debug("cPairs = [ " + cPairs.join(", ") + " ]");

    var paletteSize = 0;
    while(true) {
        palette += randomChoice(cPairs);
        palette += blank[0] + blank[0];
        paletteSize += 1;
//        if(Math.random() < 0.125) break;
//        if(Math.random() < 0.25 || paletteSize >= tileCount * 2) break;
//        if(Math.random() < 0.06) break;
        if(Math.random() < 0.02) break;
    }

	debug("paletteSize = " + paletteSize);

	// Pack a (semirandom) bunch of extra blanks on the end --------------------

    for(var i = randomInt(paletteSize) * 3; i >= 0; i -= 1) {
		palette += blank[0];
	}

	//--------------------------------------------------------------------------
	// Finally, we build the tiles themselves. As it stands, this is very
	// simple: generate tileCount tiles by filling them with randomly selected
	// characters from the palette, rejecting results that are blank or have
	// already been generated.
	//--------------------------------------------------------------------------

    var nIter = 0;

    while(true) {

		nIter       += 1;
		var tileSet = [ ];  // array of generated tiles

        for(var i = 0; i < tileCount; i++) {

			while(true) {

				for(var tile = '', j = 0; j < sides; j++)
                    tile += randomChoice(palette);

                if(tile == blank)
					continue;

				//--------------------------------------------------------------
				// This was already commented out in the original, and I haven't
				// yet had time to figure out what it does.
				//--------------------------------------------------------------

                /*function testKnottiness(chars) {
                    var tileCount = 0;
                    for(var i = 0; i < sides; i++)
                        if(chars.search(tile[i]) != -1)
                            tileCount += 1;
                    return tileCount == 0 || tileCount == 2 || tileCount == 6;
                }
                if(drawTilesKnotted) {
                    if(!testKnottiness('abcd')) continue;
                    if(!testKnottiness('ABCD')) continue;
                    if(!testKnottiness('1234')) continue;
                }*/

				//--------------------------------------------------------------
				// Repeat generation loop if this tile already exists.
				// Otherwise, it's good: push it into the tileSet and generate
				// the next one.
				//--------------------------------------------------------------
			
                tile = normalize(tile);

				if(listContains(tileSet, tile))
					continue;
                else
					break;
				
            }

            tileSet.push(tile);

        }


		//======================================================================
		// This is an interesting set of tests that should surely be made
		// optional in the future: tile sets are rejected if there are not
		// equal numbers of any pair of polar connectors that appear.
		//======================================================================

		if(useTestCharge) {

			var badCharge = false;
			for(var i = 0; i < polarPair.length; i++) {
				if(!testCharge(tileSet, polarPair[i][0], polarPair[i][1])) {
					badCharge = true;
					break;
				}
			}

			if(badCharge)
				continue;

		}

		

		//======================================================================
		// If we've tried and failed to come up with a working tile set after
		// sixteen tries, give up. (Seriously?)
		//======================================================================

        if(nIter >= 16)
			break;

		//=====================================================================
		// This is also currently a bit of a mystery. Removing it entirely
		// does NOT break the program. What it *seems* to be doing is rejecting 
		// tilesets that do not all match a few characters at the beginning of 
		// the palette, presumably to make sure that the tiles are broadly 
		// compatible. I think this is a suboptimal approach and will likely 
		// replace it. Or possibly the entire random generation system.
		//=====================================================================

		if(anyBadTest) {

			for(var anyBad = false, i = 0; i < palette.length && !anyBad; i++) {  // for each character in the palette

				if(palette[i] == '_' || palette[i] == '-')                        // skip blanks
					continue;

				for(var any = false, j = 0; j < tileCount && !any; j++) {         // for each tile

					for(var k = 0; k < sides && !any; k++) {                      // for each side

						if(tileSet[j][k] == palette[i]) {
							any = true;                                           // if the side matches a palette character, we're good
							break;
						}
					}
				}
				if(!any)
					anyBad = true;

			}
			if(anyBad)
				continue;
			else
				break;

		}

    } // end of tile generation outer loop

    document.getElementById('textfield').value = tileSet.join(", ");
    init();
    randomizing = true;

}


//##############################################################################
//# Returns true if the supplied tile set contains equal numbers of plus and
//# minus, which are the members of a polarized pair, e.g., "A" and "a", or
//# "C" and "c".
//##############################################################################

function testCharge(tileSet, plus, minus) {

	var anyPlus  = false;
	var anyMinus = false;

	for(var i = 0; i < tileSet.length; i++) {
		var charge = 0;

		for(var j = 0; j < sides; j++) {
			if(tileSet[i][j] == plus)
				charge++;
			if(tileSet[i][j] == minus)
				charge--;
		}

		if(charge > 0)
			anyPlus = true;
		if(charge < 0)
			anyMinus = true;
	}

	return anyPlus == anyMinus;
}


//##############################################################################
//# If debug is true, emits the message to the Javascript console.
//##############################################################################

function debug(msg) {
	if(debug)
		console.log(msg);
}


//##############################################################################
//# Extracts the textual SVG code from the generated image.
//##############################################################################

function SVGemit() {
    var text = '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" xmlns:ev="http://www.w3.org/2001/xml-events">\n';

    function dump(element) {
        text += '<' + element.tagName;

        for(var j = 0; j < element.attributes.length; j++)
            text += ' ' + element.attributes[j].name + '="' + element.attributes[j].value + '"';

        if(element.hasChildNodes()) {
            text += '>\n';
            for(var j = 0; j < element.childNodes.length; j++)
                dump(element.childNodes[j]);
            text += '</' + element.tagName + '>\n';
        } else {
            text += '/>\n';
        }
    }

    for(var i = 0; i < svg.childNodes.length; i++)
        dump(svg.childNodes[i]);

    text += '</svg>';

	return text;
}


//##############################################################################
//# Opens a new window with the contents of the generated SVG (FIXME: should
//# trigger a download.) If welded is true, a welded SVG is produced, possibly
//# after jiggering the display and the rejiggering it afterward.
//##############################################################################

function SVGsave(welded) {
	var svg;

	if(welded) {
		var restart    = false;
		var colorClick = false;
		var knotClick  = false;

		if(uiState == "running") {
			$("#btnPause").trigger("click");
			restart = true;
		}
		if(drawTilesColored) {
			$("#coloredCheck").trigger("click");
			colorClick = true;
		}
		if(drawTilesKnotted) {
			$("#knottedCheck").trigger("click");
			knotClick = true;
		}

		svg = SVGemit();

		if(knotClick)
			$("#knottedCheck").trigger("click");
		if(colorClick)
			$("#coloredCheck").trigger("click");
		if(restart)
			$("#btnPause").trigger("click");

		//svg = SVGweld(svg)

		SVGweld(svg)

	} else {

		svg = SVGemit();
		SVGsaveFinalize(svg);

	}
	

}


//##############################################################################
//# Prompts the user for a filename and saves the generated SVG. If wrap is
//# true, svg is output from pathsReassemble, so it needs the appropriate SVG
//# wrapper added first.
//##############################################################################

function SVGsaveFinalize(svg, wrap) {

	if(wrap) {
		svg = "<svg xmlns='http://www.w3.org/2000/svg' xmlns:xlink='http://www.w3.org/1999/xlink' xmlns:ev='http://www.w3.org/2001/xml-events'>\n"
        + "<path d='" + svg + "' fill='#DDD' stroke='black' stroke-width='1' fill-rule='evenodd' id='536788996' />\n"
        + "</svg>\n";
	}

	var blob = new Blob([svg], {type: "image/svg+xml;charset=utf-8"});

	var filename = "";
	while(filename.length == 0) {
		filename = prompt("Enter a name for your file:", "diagrammata.svg");
		if(filename === null)
			return;
		filename = filename.replace(/^\s+/, "").replace(/\s+$/, "");
	}

	saveAs(blob, filename);
}


//##############################################################################
//# Opens a separate window with the SVG content ready to save.
//##############################################################################

function makeSaveable() {

	var text = SVGemit();

    var document2 = window.open('', '_blank').document;
    document2.open();
    document2.writeln('<pre id="area"></pre>');
    document2.close();
    document2.getElementById('area').textContent = text;
}


//##############################################################################
//# Initial page setup. Scales components, preloads form elements, and parses
//# any GET arguments in the URI.
//##############################################################################

function setup() {
	resizeHandler(); 
    displayTileSize();

    var ps = document.getElementById("presets");
    if(ps) {
		var padLength = presets.length.toString().length;
        for(var i = 0; i < presets.length; i++) {
            var opt = new Option(pad((i + 1), padLength) + ". " + presets[i][0], presets[i][1]);
            ps.add(opt, null);
        }
    }

	var sc = document.getElementById("f_tile_size");
	if(sc) {
		for(var i = 1; i < 51; i++) {
			var opt = new Option(i+"x", i);
			sc.add(opt, null);
		}
		sc.selectedIndex = 11;
	}

    var parameters = document.URL.split('?')[1];
    if(parameters != null) {
        parameters = parameters.split('#')[0].replace(/&amp;/g, ", ");
        $("#textfield").val(parameters);

		// FIXME: Intelligent behavior for preset dropdown

    } else {
		$("#textfield").val(ps.value);
	}

	uiUpdate(true);
	updateParams();

	$("#url_link").width($("#welded_svg_link").width());
	$("#raw_svg_link").width($("#welded_svg_link").width());

	function pad(n, p) {
		n = n.toString();
		while(n.length < p)
			n = "0" + n;
		return n;
	}

	$(".helplink").leanModal();

}


//##############################################################################
//# Called when the browser window is resized.
//##############################################################################

function resizeHandler() {
	$("#svgOuter").height( $(window).height() - 80 );
	$("#svgOuter").css("width", "10px");
	$("#textfield").css("width", "10px");
	$("#svgOuter").css("width", "100%");
	$("#textfield").width( $("#svgOuter").outerWidth());
}


//##############################################################################
//# Updates the UI controls after a state change. If startup is defined, then 
//# this is the first call and we need to set up event handlers.
//##############################################################################

function uiUpdate(startup) {
	
	if(startup) { 

		$(window).on('resize', resizeHandler);
		resizeHandler();

		$("#btnStart").on("click", function() { init(); uiUpdate(); });
		$("#btnRandomize").on("click", function() { randomize(); uiUpdate(); });

		$("#btnPause").on("click", function() {
			if(intervalId == null && uiState == "paused") {
				intervalId = window.setInterval("iterate()", 10)
				uiState = "running";
			} else {
				stop();
				uiState = "paused";
			}
			uiUpdate();
		});

		$("#url_link").on("click", makeURL);
		$("#raw_svg_link").on("click", function() { SVGsave(false); });
		$("#welded_svg_link").on("click", function() { SVGsave(true); });
	}

	switch(uiState) {

		case "running":
			$("#btnStart").html("Restart");
			$("#btnPause").html("Pause");
			$("#btnPause").removeClass("disabled");
			break;

		case "paused":
			$("#btnStart").html("Restart");
			$("#btnPause").html("Continue");
			$("#btnPause").removeClass("disabled");
			break;

		case "finished":
			$("#btnStart").html("Start");
			$("#btnPause").html("Pause");
			$("#btnPause").addClass("disabled");
			break;

		default:
			break;
	}


}


//##############################################################################
//# Called to update the display of the global tileSize value in the
//# f_tile_size field.
//##############################################################################

function displayTileSize() {
    var ts = document.getElementById("f_tile_size");
    if(ts)
        ts.value = tileSize;
}


//##############################################################################
//# This is the onchange handler for the f_tile_size field. Checks to make sure
//# the entered value is legal. If so, truncates it to two decimals and sets the
//# global tileSize variable and calls init. Otherwise, restores the current
//# tileSize value and issues an alert.
//##############################################################################

function updateTileSize() {
    var ts = document.getElementById("f_tile_size");
    if(!ts) { return; }
    tileSize = parseFloat(ts.value);
    redraw();
}


//##############################################################################
//# Toggles drawing of colored tiles on and off.
//##############################################################################

function toggleColored() {
	drawTilesColored = !drawTilesColored;
	if(drawTilesColored) {
		$("#color_button").attr("src", "img/color_on.png");
	} else {
		$("#color_button").attr("src", "img/color_off.png");
	}
	redraw();
}


//##############################################################################
//# Toggles drawing of knotwork on and off.
//##############################################################################

function toggleKnotted() {
	drawTilesKnotted = !drawTilesKnotted;
	if(drawTilesKnotted) {
		$("#kw_button").attr("src", "img/kw_on.png");
	} else {
		$("#kw_button").attr("src", "img/kw_off.png");
	}
	redraw();
}


//##############################################################################
//# High-level wrapper function that takes the SVG output of a run and converts
//# it into a single compound curve, eliminating the joints between tiles. The
//# output is another complete SVG object.
//##############################################################################

function SVGweld(svg) {
	var svgObj = SVGparse(svg);
	var frag   = pathsFragment(svgObj);

	pbarCreate("Welding SVG tiles...");

	var garf   = pathsReassemble(frag);
}


//##############################################################################
//# Parses the raw SVG output into a form that is easier to perform shape
//# merging upon. It is presently full of assumptions about the exact form of
//# the original SVG output, so tinkering upstream is almost guaranteed to break
//# things down here.
//#
//# This presently includes a workaround for an upstream bug that results in
//# the needless repetition of identical points in the path.
//##############################################################################

function SVGparse(svg) {
    var lines = svg.split("\n");

    var svgObj = {
        header: null,
        path:   [ ]
    };

    for(var i in lines) {
        if(lines[i].substr(1, 3) == "svg") {
            svgObj.header = lines[i];
        } else if(lines[i].substr(1, 4) == "path") {
            svgObj.path.push({ raw: lines[i], points: [ ], translate: null, id: null, fill: null });
        }
    }

    var m;
    for(var i in svgObj.path) {
        var line = svgObj.path[i].raw.split(/\s+/);
		for(var l = 0; l < line.length; l++) {
			var f = parseFloat(line[l]);
			if(!isNaN(f))
				line[l] = f;
		}

        for(var j = 0; j < line.length; j++) {
            var curtok = line[j];

            if(curtok == "d=\"M" || curtok == "d='M") {
				var start = { };
				start.x = parseFloat(line[j+1]);
				start.y = parseFloat(line[j+2]);
				svgObj.path[i].start = start;
                j += 2;
            } else if(curtok == "C") {
                svgObj.path[i].points.push({ type: "curve", cx1: parseFloat(line[j+1]), cy1: parseFloat(line[j+2]), cx2: parseFloat(line[j+3]), cy2: parseFloat(line[j+4]) });
                svgObj.path[i].points.push({ type: "point", x: parseFloat(line[j+5]), y: parseFloat(line[j+6]) });
                j += 6;
            } else if(curtok == "L") {
                svgObj.path[i].points.push({ type: "line" });
                svgObj.path[i].points.push({ type: "point", x: parseFloat(line[j+1]), y: parseFloat(line[j+2])});
                j += 2;
            } else if(m = curtok.match(/fill=[\'\"]([^\'\"]+)[\'\"]/)) {
                svgObj.path[i].fill = m[1];
            } else if(m = curtok.match(/id=[\'\"]([^\'\"]+)[\'\"]/)) {
                svgObj.path[i].id = m[1];
            } else if(m = curtok.match(/translate\(([-0-9\.]+),([-0-9\.]+)/)) {
                svgObj.path[i].translate = { x: parseFloat(m[1]), y: parseFloat(m[2]) };
            }
        }
        delete svgObj.path[i].raw;
    }

    // Apply translations ------------------------------------------------------

    for(var i in svgObj.path) {

		svgObj.path[i].start.x += svgObj.path[i].translate.x;  // FIXME: BUG TRIGGERED HERE IN KNOTWORK MODE -- probably due to SVG parsing errors upstream
		svgObj.path[i].start.y += svgObj.path[i].translate.y;

        for(var j in svgObj.path[i].points) {

            if(svgObj.path[i].translate == undefined)
                continue;

            var pt = svgObj.path[i].points[j];

            if(pt.type == "point") {

				pt.x += svgObj.path[i].translate.x;
                pt.y += svgObj.path[i].translate.y;

            } else if(pt.type == "curve") {

				pt.cx1 += svgObj.path[i].translate.x;
                pt.cy1 += svgObj.path[i].translate.y;
                pt.cx2 += svgObj.path[i].translate.x;
                pt.cy2 += svgObj.path[i].translate.y;

            }
        }
        delete svgObj.path[i].translate;
    }

	// Duplicate point purge ---------------------------------------------------

    for(var i in svgObj.path) {
        var pts = svgObj.path[i].points;
        var tmp = [ pts[0], pts[1] ];

        for(var j = 2; j < pts.length; j += 2) {

            if(pts[j+1].x == pts[j-1].x && pts[j+1].y == pts[j-1].y) {
                continue;
            }
            tmp.push(pts[j]);
            tmp.push(pts[j+1]);

        }

        svgObj.path[i].points = tmp;

    }

    return svgObj;
}


//##############################################################################
//# Takes an svgObj from SVGparse and converts it into a list of segments. Each
//# segment consists of a left and a right end, plus left and right control
//# points if the segment is a cubic curve. Segments are reoriented so that
//# their lower end points left, and then they are sorted. Lower, in this case,
//# means a lower lx, and in the event of a tie, ly, rx, and ry are considered.
//# Duplicate segments are eliminated.
//##############################################################################

function pathsFragment(svgObj) {
    var frag = [ ], result = [ ];

    // Create list of fragments ------------------------------------------------

    for(var pa in svgObj.path) {

		var path = svgObj.path[pa];
        var lastX = parseFloat(path.start.x).toFixed(prec);
        var lastY = parseFloat(path.start.y).toFixed(prec);

        for(var p = 0; p < path.points.length; p += 2) {

            var segment = { }; 
            segment.lx  = lastX;
            segment.ly  = lastY;
            segment.rx  = parseFloat(path.points[p+1].x).toFixed(prec);
            segment.ry  = parseFloat(path.points[p+1].y).toFixed(prec);

            if(path.points[p].type == "curve") {

                segment.lcx = parseFloat(path.points[p].cx1).toFixed(prec);
                segment.lcy = parseFloat(path.points[p].cy1).toFixed(prec);
                segment.rcx = parseFloat(path.points[p].cx2).toFixed(prec);
                segment.rcy = parseFloat(path.points[p].cy2).toFixed(prec);

            }

            lastX = segment.rx;
            lastY = segment.ry;

            segmentReorient(segment);
            frag.push(segment);

        }

    }

    // Sort the list and strip duplicates --------------------------------------

    frag.sort(segmentSort);
    result.push(frag[0]);

    for(var f = 1; f < frag.length; f++) {
        if(frag[f].lx != frag[f-1].lx || frag[f].ly != frag[f-1].ly || frag[f].rx != frag[f-1].rx || frag[f].ry != frag[f-1].ry) {
            result.push(frag[f]);
        } else {
            result.pop();
        }
    }

    return result;
}


//##############################################################################
//# Given the result from pathsFragment, assembles them into a compound path,
//# which it then returns as the contents of the d attribute of an SVG path
//# element.
//#
//# To avoid timeouts, this gets called as timeout function which continuously
//# adjusts its own timing to repeat every welder.refresh milliseconds. On
//# the first iteration, it is called with the output from pathsFragment(). On
//# the last iteration, it generates the final SVG and calls SVGsaveFinalize
//# to dispose of it.
//##############################################################################

function pathsReassemble(frag) {

	//==========================================================================
	// On the first call, we set up the global welder state object, fire up the
	// deferred routine timer, and exit;
	//==========================================================================

	if(frag) {
		welder.frag       = frag;
		welder.subpath    = [ [ ] ];
		welder.offset     = 0;
		welder.initLength = frag.length;
		welder.jobSize    = 200;
		
		pbarUpdate(welder.initLength, 0);

		welder.timerId    = setTimeout(pathsReassemble, 1);
		return;
	}


	//==========================================================================
	// On the second and subsequent calls, jobSize fragments are welded, after
	// which the iteration timing is calculated and a new jobSize value is
	// calculated to more closely approximated the desired refresh length.
	//==========================================================================

	var start   = new Date().getTime();
	var jobSize = welder.jobSize;

    do {
        if(welder.subpath[welder.offset].length == 0) {
            welder.subpath[welder.offset].push(welder.frag.pop());
        }
        var tail = welder.subpath[welder.offset].length - 1;
        var cs = findNeighbor(welder.subpath[welder.offset][tail], welder.frag);
        if(cs == null && welder.frag.length) {
            welder.offset++;
            welder.subpath[welder.offset] = [ ];
        } else if(cs != null) {
            welder.subpath[welder.offset].push(cs);
        }

    } while(--jobSize && welder.frag.length);

	if(welder.frag.length) {

		pbarUpdate(welder.initLength, welder.initLength - welder.frag.length);

		var finish   = new Date().getTime();
		var fragTime = (finish - start) / welder.jobSize;
		welder.jobSize = Math.round(welder.refresh / fragTime) + 1;

		debug("Welder iteration: " + welder.frag.length + " fragments remaining.");

		welder.timerId = setTimeout(pathsReassemble, 1);
		return;
	}

	//==========================================================================
	// When we arrive here, reassembly is done, and all we have to do is gener-
	// ate the final SVG content and save it.
	//==========================================================================

    var result = [ ];

    for(var offset in welder.subpath) {
        var sp     = welder.subpath[offset];
        var points = [ ];

        for(var s in sp) {
            points.push([sp[s].lx, sp[s].ly]);
        }

        points.push([sp[s].lx, sp[s].ly]);

		// Push instructions onto result stack ---------------------------------

        sp = welder.subpath[offset];
        result.push("M " + sp[0].lx + " " + sp[0].ly);

        for(var p in sp) {

            if(sp[p].lcx == undefined) {
                result.push("L " + sp[p].rx + " " + sp[p].ry);
            } else {
                result.push("C " + sp[p].lcx + " " + sp[p].lcy + " " + sp[p].rcx + " " + sp[p].rcy + " " + sp[p].rx + " " + sp[p].ry);
            }
        }
        result.push("z");
    }

	pbarClose();

    SVGsaveFinalize(result.join(" "), true);
}


//##############################################################################
//# Given the current subpath, cs, and the list of remaining fragments, frag,
//# extracts and returns a neighbor, or if none is found, null. The orientation
//# of the neighbor is corrected if necessary.
//##############################################################################

function findNeighbor(cs, frag) {
    var hit = null;

    for(var ns in frag) {

        if(cs.rx == frag[ns].lx && cs.ry == frag[ns].ly) {

            hit = frag.splice(ns, 1);
			hit = hit.pop();
            break;

        } else if(cs.rx == frag[ns].rx && cs.ry == frag[ns].ry) {

            hit = frag.splice(ns, 1);
			hit = hit.pop();
            segmentReorient(hit, true);
            break;

        } 

    }

    if(hit == null) {
        return null;
    } else {
        return hit;
    }
}


//##############################################################################
//# Reorients a segment left-low if flip is undefined. If flip is defined, it
//# simply reverses the orientation. The change is made in place using the
//# reference s.
//##############################################################################

function segmentReorient(s, flip) {
    if(flip == undefined && ((s.lx < s.rx) || (s.lx == s.rx && s.ly < s.ry)))
        return;

    var x = s.lx;
    var y = s.ly;
    s.lx  = s.rx;
    s.ly  = s.ry;
    s.rx  = x;
    s.ry  = y;

    if(s.lcx == undefined)
        return;

    x     = s.lcx;
    y     = s.lcy;
    s.lcx = s.rcx;
    s.lcy = s.rcy;
    s.rcx = x;
    s.rcy = y;
}


//##############################################################################
//# Sort function for segments in ascending order by lx, ly, rx, ry.
//##############################################################################

function segmentSort(a, b) {
    if(a.lx != b.lx) {
        return a.lx - b.lx;
    } else if(a.ly != b.ly) {
        return a.ly - b.ly;
    } else if(a.rx != b.rx) {
        return a.rx - b.rx;
    } else if(a.ry != b.ry) {
        return a.ry - b.ry;
    } else {
        return 0;
    }
}


//##############################################################################
//# Updates various internal parameters and associated UI elements when some UI
//# control changes. This is a catch-all currently and will eventually be
//# refactored as the control set stabilizes.
//##############################################################################

function updateParams() {
	
	//==========================================================================
	// Maximum iterations. Can change during iteration.
	//==========================================================================

	var tmp = parseInt($("#maxIter").val());
	if(isNaN(tmp) || tmp < 0) {
		$("#maxIter").addClass("invalid");
	} else {
		maxIter = tmp;
		$("#maxIter").removeClass("invalid");
	}

	//==========================================================================
	// Workspace rotation. Can change during iteration.
	//==========================================================================
/*
	var tmp = parseFloat($("#rotation").val());
	if(isNaN(tmp)) {
		$("#rotation").addClass("invalid");
	} else {
		rotation = tmp;
		// FIXME: adjust global rotation transform
		$("#rotation").removeClass("invalid");
	}
*/
	//==========================================================================
	// Clock overlay display. Can change during iteration.
	//==========================================================================

	showClock = $("#showClock").prop("checked");
	if(showClock) {
		$("#clockDisplay").css("display", "block");
		updateClock();
	} else {
		$("#clockDisplay").css("display", "none");
	}

	//==========================================================================
	// Simple toggles and values
	//==========================================================================

	useTestCharge = $("#useTestCharge").prop("checked");  // enforce testCharge during randomization
	anyBadTest    = $("#anyBadTest").prop("checked");  // enforce testCharge during randomization
	
	fixedSides    = $("#fixedSides").val();               // randomization geometry
	if(fixedSides != "random")
		fixedSides = parseInt(fixedSides);

	baseTileCount = parseInt($("#baseTileCount").val());

	//==========================================================================
	// The application of polar and nonpolar counts is, ah, a little hackish and
	// makes a number of assumptions that I'm sure I will regret later.
	//==========================================================================

	polarCount    = $("#polarCount").val();
	
	var enabled = polarCount * 2;
	for(var c in dcon) {
		if(dcon[c].visible && dcon[c].polarity != 0) {
			if(enabled > 0)
				dcon[c].weight = 1;
			else
				dcon[c].weight = 0;
			enabled--;
		}
	}
	
	nonpolarCount = $("#nonpolarCount").val();

	var enabled = nonpolarCount;
	for(var c in dcon) {
		if(dcon[c].visible && dcon[c].polarity == 0) {
			if(enabled > 0)
				dcon[c].weight = 1;
			else
				dcon[c].weight = 0;
			enabled--;
		}
	}
}


//##############################################################################
//# Updates the clock. We do *not* check whether the clock is displayed here, so
//# the calling routine must perform that check.
//##############################################################################

function updateClock() {
	if(maxIter)
		$("#clockDisplay").html(clock + " of " + maxIter);
	else
		$("#clockDisplay").html(clock);
}


//##############################################################################
//# These three functions (which should eventually be class methods) are the
//# first stab at providing a modal, uninterruptible progress bar.
//##############################################################################

//==============================================================================
// Opens the modal progress bar dialog box. The subject is placed over the 
// progress bar itself.
//==============================================================================

function pbarCreate(subject) {
	$("#pbar_subject").html(subject);
	$("#pbar_inner").css("width", "0%");
	$("#pbar_value").html("...");
	$("#generic_progress_trigger").trigger("click");
}

//==============================================================================
// Updates the progress bar. The arguments are the total number of elements and
// the elements that are already done so far.
//==============================================================================

function pbarUpdate(total, done) {
	var pct = Math.round(100 * (done / total)) + "%";
	$("#pbar_inner").css("width", pct);
	$("#pbar_value").html(pct + " complete");
}

//==============================================================================
// Closes the progress bar dialog box.
//==============================================================================

function pbarClose() {
	$("#lean_overlay").trigger("click");
}
